//
//  RSKitTests.m
//  RSKitTests
//
//  Created by Max Lansing on 1/29/14.
//  Copyright (c) 2014 Retention Science. All rights reserved.
//

#import <XCTest/XCTest.h>
#import "RSKit.h"
#import "RSClient+Private.h"

@interface RSKitTests : XCTestCase

@end

@implementation RSKitTests

-(id)initWithInvocation:(NSInvocation *)anInvocation {
    self = [super initWithInvocation:anInvocation];
    [RSClient initializeWithSiteId:@"foo123"];
    return self;
}



- (void)setUp
{
    [super setUp];
    
    id foo = [[RSClient sharedClient] initWithSiteId:@"site12345"];
    #pragma unused(foo)

    [RSClient sharedClient].environment = RSKitEnvironmentTest;
    [RSClient enableDebugLogs];
}

- (void)tearDown
{
    // Put teardown code here. This method is called after the invocation of each test method in the class.
    [super tearDown];
}

- (void)testSharedClient {
    XCTAssertNotNil([RSClient sharedClient], @"sharedClient not initialized.");
}

- (void)testSiteId {
    XCTAssertEqual([RSClient sharedClient].siteId, @"site12345", @"siteId not set correctly");
}

- (void)testSession {
    NSTimeInterval expirationInterval = 60;
    [[RSClient sharedClient] setSessionId:@"session12345" expiresIn:expirationInterval];
    XCTAssertEqualObjects([RSClient sharedClient].sessionId, @"session12345", @"sessionId not set correctly.");
    NSDate * sessionExpiresAt = [RSClient sharedClient].sessionIdExpiresAt;
    XCTAssert([sessionExpiresAt compare:[NSDate date]] == NSOrderedDescending, @"session expiration should be greater than now.");
    XCTAssert([sessionExpiresAt compare:[NSDate dateWithTimeIntervalSinceNow:expirationInterval]] == NSOrderedAscending, @"session expiration should be less than now + expiration interval");
}

- (void)testSessionPersistence {
    NSTimeInterval expirationInterval = 60;
    [[RSClient sharedClient] setSessionId:@"session321" expiresIn:expirationInterval];
    [[RSClient sharedClient] clearSession];
    id foo = [[RSClient sharedClient] initWithSiteId:@"site12345"];
#pragma unused(foo)
    
    XCTAssertEqualObjects([RSClient sharedClient].sessionId, @"session321", @"sessionId not set correctly.");
    NSDate * sessionExpiresAt = [RSClient sharedClient].sessionIdExpiresAt;
    XCTAssert([sessionExpiresAt compare:[NSDate date]] == NSOrderedDescending, @"session expiration should be greater than now.");
    XCTAssert([sessionExpiresAt compare:[NSDate dateWithTimeIntervalSinceNow:expirationInterval]] == NSOrderedAscending, @"session expiration should be less than now + expiration interval");
}

- (void)testTrack {
    [RSClient sharedClient].userId = @"user12345";
    [[RSClient sharedClient] setSessionId:@"session12345" expiresIn:60];
    
    [[RSClient sharedClient] track:@"foo"
                        properties:@{@"proppy": @12345,
                                     @"nested": @{@"nestee": @[@1, @2, @3]}}];
    
    /* just waits until the above method is cleared off the serial queue */
    [[RSClient sharedClient] dispatchBackgroundAndWait:^{}];
    
    NSInteger messageCount = [RSClient sharedClient].messageQueue.count;
    XCTAssertEqual(messageCount, 1, @"messageQueue should contain one message, count was %d instead", messageCount);
    
    NSDictionary * payload = [RSClient sharedClient].messageQueue.firstObject;
    
    XCTAssertEqual(payload[@"action"], @"foo", @"payload action should be foo.");
    XCTAssertEqual([payload[@"proppy"] integerValue], 12345, @"payload property proppy should equal 12345");
    NSInteger nestedPayloadCount = [payload[@"nested"][@"nestee"] count];
    XCTAssertEqual(nestedPayloadCount, 3, @"nested payload array should be count 3");
    
    NSDictionary * deviceInfo = payload[@"device_info"];
    XCTAssert([deviceInfo[@"os"] isEqualToString:@"iPhone OS"], @"device info OS should equal iphone OS");
    
    XCTAssertEqual(payload[@"user_id"], @"user12345", @"user_id should equal user12345");
    XCTAssertEqual(payload[@"session_id"], @"session12345", @"session_id should equal session12345");
    
    
    [[RSClient sharedClient] track:@"foo2"
                        properties:@{@"proppy": @12345,
                                     @"nested": @{@"nestee": @[@1, @2, @3]}}];
    
    [[RSClient sharedClient] dispatchBackgroundAndWait:^{}];
    
    messageCount = [RSClient sharedClient].messageQueue.count;
    XCTAssertEqual(messageCount, 2, @"messageQueue should contain one message, count was %d instead", messageCount);
    NSDictionary * payload2 = [RSClient sharedClient].messageQueue.lastObject;

    XCTAssertEqual(payload2[@"action"], @"foo2", @"payload2 action should be foo2.");


}

- (void)testFlush {
    dispatch_semaphore_t flushSema = dispatch_semaphore_create(0);
    
    [[RSClient sharedClient] setFlushCompletionBlock:^{
        dispatch_semaphore_signal(flushSema);
    }];
    
    for (int i=0; i<RS_MIN_BATCH_SIZE+1; i++) {
        [[RSClient sharedClient] track:@"foo"
                            properties:nil];
    }

    dispatch_semaphore_wait(flushSema, DISPATCH_TIME_FOREVER);
    dispatch_release(flushSema);
    
    int messageCount = [RSClient sharedClient].messageQueue.count;
    XCTAssertEqual(messageCount, 1, @"messageQueue should contain one message after flush, count was %d instead", messageCount);
}

- (void)testMessageQueuePersistence {
    int queueCount = RS_MIN_BATCH_SIZE-1;
    for (int i=0; i<queueCount; i++) {
        [[RSClient sharedClient] track:@"foo"
                            properties:nil];
    }

    [[RSClient sharedClient] applicationWillTerminate];
    [[RSClient sharedClient] dispatchBackgroundAndWait:^{}];
    [[RSClient sharedClient] clearMessageQueue];

    id foo = [[RSClient sharedClient] initWithSiteId:@"site12345"];
    #pragma unused(foo)
   
    int messageCount = [RSClient sharedClient].messageQueue.count;
    XCTAssertEqual(messageCount, queueCount, @"messageQueue should contain %d after init, count was %d instead", queueCount, messageCount);
}

- (void)testBackgroundFlush {
    dispatch_semaphore_t flushSema = dispatch_semaphore_create(0);
    
    [[RSClient sharedClient] setFlushCompletionBlock:^{
        dispatch_semaphore_signal(flushSema);
    }];
    
    for (int i=0; i<RS_MIN_BATCH_SIZE-1; i++) {
        [[RSClient sharedClient] track:@"foo"
                            properties:nil];
    }
    
    [[RSClient sharedClient] applicationDidEnterBackground];
    
    dispatch_semaphore_wait(flushSema, DISPATCH_TIME_FOREVER);
    dispatch_release(flushSema);
    
    int messageCount = [RSClient sharedClient].messageQueue.count;
    XCTAssertEqual(messageCount, 0, @"messageQueue should be empty after flush, count was %d instead", messageCount);
}





@end
