//
//  RSUtils.m
//  RSKit
//
//  Created by Max Lansing on 1/29/14.
//  Copyright (c) 2014 Retention Science. All rights reserved.
//

#import "RSUtils.h"


static BOOL kAnalyticsLoggerShowLogs = NO;

/* Gives an NSURL, located in the ApplicationSupport directory, for an arbitrary filename. */

NSURL *RSURLForFilename(NSString *filename) {
    NSString *appSupportDir = [NSSearchPathForDirectoriesInDomains(NSApplicationSupportDirectory, NSUserDomainMask, YES) lastObject];
    //Create app support dir if there isn't one yet.
    if (![[NSFileManager defaultManager] fileExistsAtPath:appSupportDir isDirectory:NULL]) {
        NSError *error = nil;
        if (![[NSFileManager defaultManager] createDirectoryAtPath:appSupportDir withIntermediateDirectories:YES attributes:nil error:&error]) {
            NSLog(@"%@", error.localizedDescription);
        }
        else {
            //exclude from icloud backups
            [[NSURL fileURLWithPath:appSupportDir]
             setResourceValue:[NSNumber numberWithBool:YES]
             forKey:NSURLIsExcludedFromBackupKey
             error:&error];
        }
    }
    NSString *path = [appSupportDir stringByAppendingPathComponent:filename];
    return [NSURL fileURLWithPath:path];
}

// Async Utils

//Creates a serial queue.
dispatch_queue_t dispatch_queue_create_specific(const char *label, dispatch_queue_attr_t attr) {
    dispatch_queue_t queue = dispatch_queue_create(label, attr);
    dispatch_queue_set_specific(queue, (const void *)queue, (void *)queue, NULL);
    return queue;
}

//Checks whether we're currently on a queue
BOOL dispatch_is_on_specific_queue(dispatch_queue_t queue) {
    return dispatch_get_specific((const void *)queue) != NULL;
}

//Dispatch the block to the queue in an appropriate manner.
void dispatch_specific(dispatch_queue_t queue, dispatch_block_t block, BOOL waitForCompletion) {
    if (dispatch_get_specific((const void *)queue)) {
        //If we're currently in a block dispatched to the queue, just run the passed-in block
        block();
    } else if (waitForCompletion) {
        //Execute the block synchronously
        dispatch_sync(queue, block);
    } else {
        dispatch_async(queue, block);
    }
}

void dispatch_specific_async(dispatch_queue_t queue, dispatch_block_t block) {
    dispatch_specific(queue, block, NO);
}

void dispatch_specific_sync(dispatch_queue_t queue, dispatch_block_t block) {
    dispatch_specific(queue, block, YES);
}

// Logging

//Setter for verbose logging flag.

void SetShowDebugLogs(BOOL showDebugLogs) {
    kAnalyticsLoggerShowLogs = showDebugLogs;
}

//Custom logging wrapper - requires that verbose logs are enabled before we log anything.

void RSLog(NSString *format, ...) {
    if (kAnalyticsLoggerShowLogs) {
        va_list args;
        va_start(args, format);
        NSLogv(format, args);
        va_end(args);
    }
}

// JSON Utils

//Coerces object of your choice into a form that can be serialized to JSON.

static id CoerceJSONObject(id obj) {
    // if the object is a NSString, NSNumber or NSNull
    // then we're good
    if ([obj isKindOfClass:[NSString class]] ||
        [obj isKindOfClass:[NSNumber class]] ||
        [obj isKindOfClass:[NSNull class]]) {
        return obj;
    }
    
    if ([obj isKindOfClass:[NSArray class]]) {
        NSMutableArray *array = [NSMutableArray array];
        for (id i in obj)
            [array addObject:CoerceJSONObject(i)];
        return array;
    }
    
    if ([obj isKindOfClass:[NSDictionary class]]) {
        NSMutableDictionary *dict = [NSMutableDictionary dictionary];
        for (NSString *key in obj) {
            if (![key isKindOfClass:[NSString class]])
                NSLog(@"warning: dictionary keys should be strings. got: %@. coercing to: %@", [key class], [key description]);
            dict[key.description] = CoerceJSONObject(obj[key]);
        }
        return dict;
    }
    
    // NSDate description is already a valid ISO8061 string
    if ([obj isKindOfClass:[NSDate class]])
        return [obj description];
    
    if ([obj isKindOfClass:[NSURL class]])
        return [obj absoluteString];
    
    // default to sending the object's description
    NSLog(@"warning: dictionary values should be valid json types. got: %@. coercing to: %@", [obj class], [obj description]);
    return [obj description];
}

static void AssertDictionaryTypes(id dict) {
    assert([dict isKindOfClass:[NSDictionary class]]);
    for (id key in dict) {
        assert([key isKindOfClass: [NSString class]]);
        id value = dict[key];
        
        assert([value isKindOfClass:[NSString class]] ||
               [value isKindOfClass:[NSNumber class]] ||
               [value isKindOfClass:[NSNull class]] ||
               [value isKindOfClass:[NSArray class]] ||
               [value isKindOfClass:[NSDictionary class]] ||
               [value isKindOfClass:[NSDate class]] ||
               [value isKindOfClass:[NSURL class]]);
    }
}

NSDictionary *CoerceDictionary(NSDictionary *dict) {
    // make sure that a new dictionary exists even if the input is null
    dict = dict ?: @{};
    // assert that the proper types are in the dictionary
    AssertDictionaryTypes(dict);
    // coerce urls, and dates to the proper format
    return CoerceJSONObject(dict);
}
