//
//  RSClient.h
//  RSKit
//
//  Created by Max Lansing on 1/29/14.
//  Copyright (c) 2014 Retention Science. All rights reserved.
//

#import <Foundation/Foundation.h>

@class CLLocationManager;

/* This macro is provided by foundation in iOS 6 and up, so we include it here in case older versions need support */
#ifndef NS_ENUM
#define NS_ENUM(_type, _name) enum _name : _type _name; enum _name : _type
#endif

/** 
Represents a runtime environmental mode for RSKit.
 */
typedef NS_ENUM(NSInteger, RSKitEnvironment) {
/**
Use the main Retention Science production API endpoint.
 */
    RSKitEnvironmentProduction,
/**
Use the Retention Science staging/sandbox API endpoint.
 */
    RSKitEnvironmentStaging,
/**
Use a localhost endpoint, for testing.
 */
    RSKitEnvironmentTest
};



/** This is the main class in the Retention Science iOS SDK.
 */


@interface RSClient : NSObject


/** @name User Identification
 */

/**
 An identifying string for the current user.
 */
@property NSString * userId;

/** @name Environments
 */

/**
 The environmental mode the SDK should run in. This will determine which API endpoint is used. The default value is RSKitEnvironmentProduction.
 @see RSKitEnvironment
 */
@property enum RSKitEnvironment environment;


/** @name Location
 */

/**
 Expose a CLLocationManager instance to the RSClient. The most recent location updates, if there are any, will be included with subsequent tracking messages. RSKit will never start or stop location updates on its own.
 @note Optional.
 */
@property CLLocationManager * locationManager;



/**
 @name Initialization
 */

/**
 Initialize the singleton RSClient instance with your Retention Science site ID.
 @param siteId The unique site ID provided to you by Retention Science.
 @warning siteId must not be nil.
 */
+ (void)initializeWithSiteId:(NSString *)siteId;


/**
 Enables verbose logs for debugging purposes.
 */
+ (void)enableDebugLogs;

/**
 @name Tracking
 */

/**
 Accessor for the singleton RSClient instance.
 @warning Make sure you have called initializeWithSiteId: before call this.
 @return The singleton RSClient instance.
 @see initializeWithSiteId:
 */
+ (instancetype)sharedClient;



/**
 Tracks an action, as well as associated properties. The tracking message will be queued and posted to the RS API when the queue is automatically flushed.
 @param action The name of the action you want to track.
 @param properties Arbitrary properties that are associated with this action.
 */
- (void)track:(NSString *)action properties:(NSDictionary *)properties;

/**
 Assigns an arbitrary session ID, as well as an expiration interval, which will be included in all subsequent tracking messages. The session ID is persisted to disk and is retained until the expiration interval is exceeded.
 
 @param sessionId The session ID, such as an offer code.
 @param interval Interval that session ID is retained for.
 */
- (void)setSessionId:(NSString *)sessionId expiresIn:(NSTimeInterval)interval;


@end
