//
//  RSClient.m
//  RSKit
//
//  Created by Max Lansing on 1/29/14.
//  Copyright (c) 2014 Retention Science. All rights reserved.
//

#import <UIKit/UIKit.h>
#include <sys/sysctl.h>
#import <AdSupport/ASIdentifierManager.h>
#import <CoreTelephony/CTCarrier.h>
#import <CoreTelephony/CTTelephonyNetworkInfo.h>
#import <CoreLocation/CoreLocation.h>

#import "RSClient.h"
#import "RSClient+Private.h"
#import "RSUtils.h"
#import "RSRequest.h"

#define RS_DISK_QUEUE_URL RSURLForFilename(@"com.retentionscience.queue.plist")
#define RS_DISK_SESSION_URL RSURLForFilename(@"com.retentionscience.session.plist")

static NSString *GenerateUUIDString() {
    CFUUIDRef theUUID = CFUUIDCreate(NULL);
    NSString *UUIDString = (__bridge_transfer NSString *)CFUUIDCreateString(NULL, theUUID);
    CFRelease(theUUID);
    return UUIDString;
}

@implementation RSClient {
}


- (RSClient *)initWithSiteId:(NSString *)siteId {
    if (self = [self init]) {
        _environment = RSKitEnvironmentProduction;
        _siteId = siteId;
        _serialQueue = dispatch_queue_create_specific("com.retentionscience.client", DISPATCH_QUEUE_SERIAL);
        _lastFlushedAt = nil;
        _flushCompletionBlock = nil;
        
        _messageQueue = [NSMutableArray arrayWithContentsOfURL:RS_DISK_QUEUE_URL];
        [[NSFileManager defaultManager] removeItemAtURL:RS_DISK_QUEUE_URL error:NULL];
        if (!_messageQueue)
            _messageQueue = [[NSMutableArray alloc] init];
        
        [self loadSession];
        
        // Attach to application state change hooks
        for (NSString *name in @[UIApplicationDidEnterBackgroundNotification,
                UIApplicationWillEnterForegroundNotification,
                UIApplicationWillTerminateNotification,
                UIApplicationWillResignActiveNotification,
                UIApplicationDidBecomeActiveNotification]) {
            NSNotificationCenter *nc = [NSNotificationCenter defaultCenter];
            [nc addObserver:self selector:@selector(handleAppStateNotification:) name:name object:nil];
        }
        
        _deviceInformation = [self getDeviceInformation];

        _flushTimer = [NSTimer scheduledTimerWithTimeInterval:RS_FLUSH_MAX_DELAY_SECONDS
                                                       target:self
                                                     selector:@selector(flush)
                                                     userInfo:nil
                                                      repeats:YES];
        
        RSLog(@"%@ initialized with site ID %@ and device info %@", self, _siteId, _deviceInformation);
        
        //give it an initial flush in case we loaded queued messages from disk
        [self flush]; 

    }
    
    return self;
}


#pragma mark - NSNotificationCenter Callback

- (void)handleAppStateNotification:(NSNotification *)note {
    RSLog(@"%@ Application state change notification: %@", self, note.name);
    static NSDictionary *selectorMapping;
    static dispatch_once_t selectorMappingOnce;
    dispatch_once(&selectorMappingOnce, ^{
        selectorMapping = @{
        UIApplicationDidEnterBackgroundNotification:
            NSStringFromSelector(@selector(applicationDidEnterBackground)),
        UIApplicationWillEnterForegroundNotification:
            NSStringFromSelector(@selector(applicationWillEnterForeground)),
        UIApplicationWillTerminateNotification:
            NSStringFromSelector(@selector(applicationWillTerminate)),
        UIApplicationWillResignActiveNotification:
            NSStringFromSelector(@selector(applicationWillResignActive)),
        UIApplicationDidBecomeActiveNotification:
            NSStringFromSelector(@selector(applicationDidBecomeActive))
                            };
    });
    SEL selector = NSSelectorFromString(selectorMapping[note.name]);
    if (selector) {
        dispatch_specific_async(_serialQueue, ^{
            NSMethodSignature * methodSignature = [self methodSignatureForSelector:selector];
            NSInvocation * invocation = [NSInvocation invocationWithMethodSignature:methodSignature];
            invocation.selector = selector;
            [invocation invokeWithTarget:self];
        });
    }
}

#pragma mark - Device Info

- (NSMutableDictionary *)getDeviceInformation
{
    NSMutableDictionary *deviceInfo = [NSMutableDictionary dictionary];
    
    // Application information
    [deviceInfo setValue:[[[NSBundle mainBundle] infoDictionary] objectForKey:@"CFBundleVersion"] forKey:@"appVersion"];
    [deviceInfo setValue:[[[NSBundle mainBundle] infoDictionary] objectForKey:@"CFBundleShortVersionString"] forKey:@"appReleaseVersion"];
    
    // Device information
    UIDevice *device = [UIDevice currentDevice];
    NSString *deviceModel = [self deviceModel];
    [deviceInfo setValue:@"Apple" forKey:@"deviceManufacturer"];
    [deviceInfo setValue:deviceModel forKey:@"deviceModel"];
    [deviceInfo setValue:[device systemName] forKey:@"os"];
    [deviceInfo setValue:[device systemVersion] forKey:@"osVersion"];
    
    // Network Carrier
    if (NSClassFromString(@"CTTelephonyNetworkInfo")) {
        id networkInfo = [[NSClassFromString(@"CTTelephonyNetworkInfo") alloc] init];
        id carrier = [networkInfo subscriberCellularProvider];
        if ([[carrier carrierName] length]) {
            [deviceInfo setValue:[carrier carrierName] forKey:@"carrier"];
        }
    }
    
    // Device ID
    if (NSClassFromString(@"ASIdentifierManager")) {
        [deviceInfo setValue:@"advertiser" forKey:@"deviceIdSource"];
        [deviceInfo setValue:[NSClassFromString(@"ASIdentifierManager") sharedManager].advertisingIdentifier.UUIDString forKey:@"deviceId"];
    } else {
        [deviceInfo setValue:@"vendor" forKey:@"deviceIdSource"];
        [deviceInfo setValue:[[UIDevice currentDevice] identifierForVendor].UUIDString forKey:@"deviceId"];
    }
    
    // Screen size
    CGSize screenSize = [UIScreen mainScreen].bounds.size;
    [deviceInfo setValue:[NSNumber numberWithInt:(int)screenSize.width] forKey:@"screenWidth"];
    [deviceInfo setValue:[NSNumber numberWithInt:(int)screenSize.height] forKey:@"screenHeight"];
    
    //Screen scale
    [deviceInfo setValue:[NSNumber numberWithFloat:[UIScreen mainScreen].scale] forKey:@"screenScale"];

    //Language
    [deviceInfo setValue:[[NSLocale preferredLanguages] objectAtIndex:0] forKey:@"language"];
    
    return deviceInfo;
}

- (NSString *)deviceModel
{
    size_t size;
    sysctlbyname("hw.machine", NULL, &size, NULL, 0);
    char result[size];
    sysctlbyname("hw.machine", result, &size, NULL, 0);
    NSString *results = [NSString stringWithCString:result encoding:NSUTF8StringEncoding];
    return results;
}


#pragma mark - Public stuff

- (void)track:(NSString *)action properties:(NSDictionary *)properties {
    NSAssert(action.length, @"%@ track requires an action name.", self);
    [self enqueueAction:action dictionary:properties];
}

- (void)setSessionId:(NSString *)sessionId expiresIn:(NSTimeInterval)interval {
    _sessionId = sessionId;
    _sessionIdExpiresAt = [NSDate dateWithTimeInterval:interval sinceDate:[NSDate date]];

    [self saveSession];
}

- (void)saveSession {
    NSDictionary * sessionDict = @{@"SessionId": _sessionId,
                                   @"SessionIdExpiresAt": _sessionIdExpiresAt};
    if (![sessionDict writeToURL:RS_DISK_SESSION_URL atomically:YES])
        RSLog(@"%@ Error saving session.", self);
}

- (void)loadSession {
    NSDictionary * sessionDict = [NSDictionary dictionaryWithContentsOfURL:RS_DISK_SESSION_URL];
    _sessionId = sessionDict[@"SessionId"];
    _sessionIdExpiresAt = sessionDict[@"SessionIdExpiresAt"];
}

#pragma mark - API stuff

- (NSDictionary *)mergePayloadForAction:(NSString *)action dictionary:(NSDictionary *)dictionary {
    if (!dictionary)
        dictionary = @{};
    NSMutableDictionary *payload = [NSMutableDictionary dictionaryWithDictionary:dictionary];
    payload[@"action"] = action;
    payload[@"site_id"] = _siteId;
    payload[@"device_info"] = _deviceInformation;
    payload[@"rsci_vid"] = [_deviceInformation objectForKey:@"deviceId"];
    payload[@"queued_at"] = [NSNumber numberWithLong:[[NSDate date] timeIntervalSince1970]];
    payload[@"guid"] = GenerateUUIDString();
    payload[@"version"] = RS_WAVE_VERSION;
    payload[@"sdk_version"] = RSKIT_VERSION;
    
    if (self.userId)
        payload[@"user_id"] = self.userId;
    
    if (self.locationManager.location) {
        _lastLocation = self.locationManager.location;
    }
    
    if (_lastLocation) {
        NSTimeInterval locationAge = [[NSDate date]
                                      timeIntervalSinceDate:_lastLocation.timestamp];
        payload[@"location"] = @{
            @"latitude": [NSNumber numberWithDouble:_lastLocation.coordinate.latitude],
            @"longitude": [NSNumber numberWithDouble:_lastLocation.coordinate.longitude],
            @"age": [NSNumber numberWithLongLong:(long long)locationAge]
        };
    }

    if (_sessionId.length &&
        [_sessionIdExpiresAt compare:[NSDate date]] == NSOrderedDescending) {
        payload[@"session_id"] = _sessionId;
    }
    
    return payload;
}

- (void)enqueueAction:(NSString *)action dictionary:(NSDictionary *)dictionary {
    NSDictionary * payload = [self mergePayloadForAction:action dictionary:dictionary];
    
    [self dispatchBackground:^{
        RSLog(@"%@ Enqueueing action: %@", self, payload);
        [_messageQueue addObject:payload];
        [self flushQueueByLengthAndTime];
    }];
}


- (void)flush {
    [self flushWithMaxSize:RS_MAX_BATCH_SIZE];
}

- (void)flushWithMaxSize:(NSUInteger)maxBatchSize {
    [self dispatchBackground:^{
        if ([_messageQueue count] == 0) {
            RSLog(@"%@ No queued API calls to flush.", self);
            return;
        } else if (_currentBatch != nil) {
            RSLog(@"%@ Already flushing a batch, not gonna flush. %@", self, _currentBatch);
            return;
        } else if ([_messageQueue count] >= maxBatchSize) {
            _currentBatch = [[_messageQueue subarrayWithRange:NSMakeRange(0, maxBatchSize)] mutableCopy];
        } else {
            _currentBatch = [NSMutableArray arrayWithArray:_messageQueue];
        }
        
        RSLog(@"%@ Flushing batch of %d events.", self, _currentBatch.count);

        [self flushNextAction];
    }];
}

- (void)flushNextAction {
    [self dispatchBackground:^{
        if (_currentBatch.count == 0) {
            [self flushDidFinish];
            return;
        }

        NSDictionary * payload = _currentBatch.firstObject;
        [self sendPayload:payload];
    }];
}

- (void)flushQueueByLengthAndTime {
    [self dispatchBackground:^{
        BOOL tooSoon = ([[NSDate date] timeIntervalSinceDate:_lastFlushedAt] < RS_FLUSH_MIN_DELAY_SECONDS);
        if (tooSoon)
            RSLog(@"%@ Skipping RS flush; too soon.", self);
        if (_messageQueue.count >= RS_MIN_BATCH_SIZE && !tooSoon) {
            [self flush];
        }
    }];
}

- (void)flushDidFinish {
    _currentBatch = nil;
    _lastFlushedAt = [NSDate date];
    [self endBackgroundTask];
    if (_flushCompletionBlock) {
        _flushCompletionBlock();
    }
}

- (void)sendPayload:(NSDictionary*)payload {
    
    NSMutableDictionary * mergedPayload = [NSMutableDictionary dictionaryWithDictionary:payload];
    
    //Calculate how long since this event was queued.
    NSTimeInterval queueWait = [[NSDate date] timeIntervalSince1970] - [payload[@"queued_at"] longValue];
    mergedPayload[@"queue_wait"] = [NSNumber numberWithLong:queueWait];
    
    NSData * payloadJSONData = [NSJSONSerialization dataWithJSONObject:CoerceDictionary(mergedPayload) options:0 error:NULL];
    NSString * payloadJSONString = [[NSString alloc] initWithData:payloadJSONData encoding:NSUTF8StringEncoding];
    NSDictionary * queryParams = @{@"wave": payloadJSONString};
    
    //Make url with get request
//    NSURL * urlWithQuery = [RSClient addQueryStringToUrl:[self remoteURL] withDictionary:queryParams];
//    NSMutableURLRequest *urlRequest = [NSMutableURLRequest requestWithURL:urlWithQuery];
    NSMutableURLRequest * urlRequest = [NSMutableURLRequest requestWithURL:[self remoteURL]];
    [urlRequest setValue:@"gzip" forHTTPHeaderField:@"Accept-Encoding"];
    urlRequest.HTTPMethod = @"POST";
    urlRequest.HTTPBody = [[RSClient queryStringFromDictionary:queryParams] dataUsingEncoding:NSUTF8StringEncoding allowLossyConversion:YES];

//    RSLog(@"%@ starting API request with URL %@", self, urlWithQuery.absoluteString);
    
    [RSRequest startWithURLRequest:urlRequest completion:^(RSRequest *request) {
       [self dispatchBackground:^{
           if (request.error) {
               RSLog(@"%@ API request had an error: %@", self, request.error);
           } else {
               RSLog(@"%@ API request success.", self);
               [_messageQueue removeObject:payload];
           }
           
           [_currentBatch removeObject:payload];
           [self flushNextAction];
       }];
    }];
}

- (NSURL *)remoteURL {
    switch (_environment) {
        case RSKitEnvironmentProduction:
            return RS_WAVE_URL_PRODUCTION;
        case RSKitEnvironmentStaging:
            return RS_WAVE_URL_STAGING;
        case RSKitEnvironmentTest:
            return RS_WAVE_URL_TEST;
    }
}

- (void)dispatchBackground:(void(^)(void))block {
    dispatch_specific_async(_serialQueue, block);
}

- (void)dispatchBackgroundAndWait:(void(^)(void))block {
    dispatch_specific_sync(_serialQueue, block);
}

#pragma mark - Background task handling


- (void)beginBackgroundTask {
    [self endBackgroundTask];
    _flushTaskID = [[UIApplication sharedApplication] beginBackgroundTaskWithExpirationHandler:^{
        [self endBackgroundTask];
    }];
}

- (void)endBackgroundTask {
    [self dispatchBackgroundAndWait:^{
        if (_flushTaskID != UIBackgroundTaskInvalid) {
            [[UIApplication sharedApplication] endBackgroundTask:_flushTaskID];
            _flushTaskID = UIBackgroundTaskInvalid;
        }
    }];
}

#pragma mark - Application State Callbacks


- (void)applicationDidEnterBackground {
    RSLog(@"%@ Beginning background task to flush RS event queue.", self);
    
    //We don't want to respect the min flush delay in this case because this is a last-ditch attempt.
    _lastFlushedAt = nil;
    
    [self beginBackgroundTask];
    [self flushWithMaxSize:100];
}

- (void)applicationWillTerminate {
    RSLog(@"%@ Writing RS message queue to disk.", self);
    
    [self dispatchBackgroundAndWait:^{
        if (_messageQueue.count)
            if (![_messageQueue writeToURL:RS_DISK_QUEUE_URL atomically:YES])
                RSLog(@"%@ Error writing message queue to disk.", self);
    }];
}

/* State change handlers below stubbed out for possible future use. */

- (void)applicationWillEnterForeground {
    
}


- (void)applicationWillResignActive {
    
}

- (void)applicationDidBecomeActive {
    
}


#pragma mark - Class Methods

static RSClient * SharedClient = nil;

//Initialize singleton RSClient instance.
+ (void)initializeWithSiteId:(NSString *)siteId {
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        SharedClient = [[self alloc] initWithSiteId:siteId];
    });
}

+ (instancetype)sharedClient {
    NSAssert(SharedClient, @"%@ sharedclient called before initializeWithSiteId", self);
    return SharedClient;
}

+ (void)enableDebugLogs {
    SetShowDebugLogs(YES);
}


//Utility method to make a query string from a dictionary.
+ (NSString*)queryStringFromDictionary:(NSDictionary *)dictionary
{
    NSMutableString * queryString = [[NSMutableString alloc] init];
    
    for (id key in dictionary) {
        NSString *keyString = [key description];
        NSString *valueString = [[dictionary objectForKey:key] description];
        
        if (queryString.length) {
            [queryString appendFormat:@"&%@=%@", [self urlEscapeString:keyString], [self urlEscapeString:valueString]];
        } else {
            [queryString appendFormat:@"%@=%@", [self urlEscapeString:keyString], [self urlEscapeString:valueString]];
        }
    }
    return [NSString stringWithString:queryString];
}


//Utility method to append a dictionary to a base URL as a GET query string.
+ (NSURL*)addQueryStringToUrl:(NSURL *)url withDictionary:(NSDictionary *)dictionary
{
    NSMutableString * urlWithQueryString = [[NSMutableString alloc] initWithString:[url absoluteString]];
    
    for (id key in dictionary) {
        NSString *keyString = [key description];
        NSString *valueString = [[dictionary objectForKey:key] description];
        
        if ([urlWithQueryString rangeOfString:@"?"].location == NSNotFound) {
            [urlWithQueryString appendFormat:@"?%@=%@", [self urlEscapeString:keyString], [self urlEscapeString:valueString]];
        } else {
            [urlWithQueryString appendFormat:@"&%@=%@", [self urlEscapeString:keyString], [self urlEscapeString:valueString]];
        }
    }
    return [NSURL URLWithString:urlWithQueryString];
}

+ (NSString*)urlEscapeString:(NSString *)unencodedString
{
    return (NSString *)CFBridgingRelease(CFURLCreateStringByAddingPercentEscapes(NULL,
            (CFStringRef)unencodedString,
            NULL,
            (CFStringRef)@"!*'\"();:@&=+$,/?%#[]% ",
            CFStringConvertNSStringEncodingToEncoding(NSUTF8StringEncoding)));

}


@end
